//
//  ALEventService.h
//  AppLovinSDK
//
//  Created by Thomas So on 2/13/19
//  Copyright © 2020 AppLovin Corporation. All rights reserved.
//

NS_ASSUME_NONNULL_BEGIN

/**
 * Service that tracks various analytical events.
 */
@interface ALEventService : NSObject

/**
 * Tracks an event without any parameters.
 *
 * @param event The name of the event to track (e.g., "page_view", "purchase").
 *
 * @discussion Use this method for simple events that do not require additional parameters.
 *             If you need to provide additional data, use the `trackEvent:parameters:` variant.
 */
- (void)trackEvent:(NSString *)event;

/**
 * Tracks an event with additional parameters.
 *
 * @param event The name of the event to track (e.g., "page_view", "purchase").
 * @param parameters A dictionary of parameter key-value pairs providing additional information about the event.
 */
- (void)trackEvent:(NSString *)event parameters:(nullable NSDictionary<NSString *, id> *)parameters;

/**
 * Tracks an event with additional parameters and options.
 *
 * @param event The name of the event to track (e.g., "page_view", "purchase").
 * @param parameters A dictionary of parameter key-value pairs providing additional information about the event.
 * @param options A dictionary of additional options to be passed up with the event. Accepted options:
 *  "dedupe_id" - A unique identifier used to uniquely identify and merge identical events occurring within a certain timeframe of each other—across both site-to-site and Axon Pixel integrations—so they’re only counted once.
 */
- (void)trackEvent:(NSString *)event
        parameters:(NSDictionary<NSString *, id> *)parameters
           options:(NSDictionary<NSString *, id> *)options;

/**
 * Tracks an in-app purchase.
 *
 * AppLovin recommends that you use one of the predefined strings provided in ALEventTypes.h for the parameter keys, when one of those strings applies
 * to the event. At a minimum, provide the following parameters: @c kALEventParameterProductIdentifierKey, @c kALEventParameterRevenueAmountKey, and
 * @c kALEventParameterRevenueCurrencyKey. If you pass a value for @c kALEventParameterStoreKitReceiptKey, AppLovin will use that value for validation.
 * Otherwise, AppLovin will collect @code +[NSBundle mainBundle] @endcode ⇒ @code -[NSBundle appStoreReceiptURL] @endcode and use it for validation.
 *
 * @param transactionIdentifier Value of the @code -[SKTransaction transactionIdentifier] @endcode property.
 * @param parameters            A dictionary that contains key-value pairs that further describe this event.
 */
- (void)trackInAppPurchaseWithTransactionIdentifier:(NSString *)transactionIdentifier parameters:(nullable NSDictionary<NSString *, id> *)parameters;

/**
 * Tracks a checkout / standard purchase.
 *
 * AppLovin recommends that you use one of the predefined strings provided in ALEventTypes.h for the parameter keys, when one of those strings applies to the
 * event. At a minimum, provide the following parameters: @c kALEventParameterProductIdentifierKey, @c kALEventParameterRevenueAmountKey, and
 * @c kALEventParameterRevenueCurrencyKey.
 *
 * @param transactionIdentifier An optional unique identifier for this transaction, generated by you. For Apple Pay transactions, AppLovin suggests that you use
 *                              the value of the @code -[PKPaymentToken transactionIdentifier] @endcode property.
 * @param parameters            A dictionary that contains key-value pairs that further describe this event.
 */
- (void)trackCheckoutWithTransactionIdentifier:(nullable NSString *)transactionIdentifier parameters:(nullable NSDictionary<NSString *, id> *)parameters;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

NS_ASSUME_NONNULL_END
