package com.applovin.mediation;

import android.content.Context;

import com.applovin.impl.mediation.MaxAdViewConfigurationImpl;
import com.applovin.mediation.ads.MaxAdView;

import androidx.annotation.NonNull;

/**
 * This class contains configurable fields for the {@link MaxAdView}.
 * 
 * NOTE: This is an abstract class rather than an interface to fix the static method not found error when minimum Android SDK is <24.
 * TODO: Return to interface when our minimum Android SDK is 24
 */
public abstract class MaxAdViewConfiguration
{
    public enum AdaptiveType
    {
        /**
         * Default type - standard banner/leader or MREC.
         */
        NONE( "none" ),

        /**
         * Adaptive ad view anchored to the screen. The MAX SDK determines the height of the adaptive ad view by invoking {@link MaxAdFormat#getAdaptiveSize(int, Context)}
         * which results in a height that ranges from 50 to 90 dp but does not exceed 15% of the screen height in the current orientation.
         */
        ANCHORED( "anchored" ),

        /**
         * Adaptive ad view embedded within scrollable content. The height can extend up to the device height in the current orientation unless you restrict it by invoking {@link Builder#setInlineMaximumHeight(int)}.
         */
        INLINE( "inline" );

        private final String label;

        AdaptiveType(String label)
        {
            this.label = label;
        }

        @NonNull
        @Override
        public String toString()
        {
            return label;
        }
    }

    /**
     * Creates a builder object for {@link MaxAdViewConfiguration}.
     * Please call {@code build()} to create a {@link MaxAdViewConfiguration} object.
     *
     * @return a {@link MaxAdViewConfiguration.Builder} object.
     */
    public static Builder builder()
    {
        return new MaxAdViewConfigurationImpl.BuilderImpl();
    }

    /**
     * @return the adaptive type of the {@link MaxAdView}. Defaults to {@link AdaptiveType#NONE}.
     */
    public abstract AdaptiveType getAdaptiveType();

    /**
     * @return the custom width, in dp, for the adaptive {@link MaxAdView}. Defaults to {@code -1}, which indicates that the width adapts to span the application window.
     * This value is only used when you set the {@link AdaptiveType} to {@link AdaptiveType#ANCHORED} or {@link AdaptiveType#INLINE}.
     */
    public abstract int getAdaptiveWidth();

    /**
     * @return the maximum height, in dp, for an inline adaptive {@link MaxAdView}. Defaults to {@code -1}, allowing the height to extend up to the device height.
     * This value is only used when you set the {@link AdaptiveType} to {@link AdaptiveType#INLINE}.
     */
    public abstract int getInlineMaximumHeight();

    /**
     * Builder class that you can instantiate to create a {@link MaxAdViewConfiguration} object before you create a {@link MaxAdView}.
     * This class contains properties that you can set to configure the properties of the {@link MaxAdView} that results from the configuration object this class builds.
     */
    public interface Builder
    {
        /**
         * Sets the {@link AdaptiveType} of the {@link MaxAdView}.
         */
        Builder setAdaptiveType(final AdaptiveType adaptiveType);

        /**
         * Sets the custom width, in dp, for the adaptive {@link MaxAdView}. Must not exceed the width of the application window.
         */
        Builder setAdaptiveWidth(final int adaptiveWidth);

        /**
         * Sets the custom maximum height, in dp, for an inline adaptive {@link MaxAdView}.
         * Must be at least 32 dp and no more than the device height in the current orientation. A minimum of 50 dp is recommended.
         */
        Builder setInlineMaximumHeight(final int inlineMaxHeight);

        /**
         * Builds a {@link MaxAdViewConfiguration} object from the builder properties' values.
         *
         * @return a {@link MaxAdViewConfiguration} object.
         */
        MaxAdViewConfiguration build();
    }
}
